﻿/* Copyright 2015 Intellica Corporation 
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at
    http://www.apache.org/licenses/LICENSE-2.0
Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/
using System;
using System.Data;
using System.Configuration;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Web.Security;
using System.Web.UI;
using System.Web.UI.HtmlControls;
using System.Web.UI.WebControls;
using System.Web.UI.WebControls.WebParts;
using System.Xml.Linq;
using ext = Ext.Net;

/// <summary>
/// US:5967 US:5959
/// Renders and handles the Patient Treatment Tree
/// </summary>
public class CTxTree
{
    protected BaseMaster m_BaseMstr { set; get; }
    
    //direct method alias
    protected string m_dmAlias { set; get; }
    protected string m_strDirectMethodPath { set; get; }

    /// <summary>
    /// Constructor to create a new instance object for the class 
    /// </summary>
    /// <param name="BaseMstr"></param>
    public CTxTree(BaseMaster BaseMstr)
	{
        m_BaseMstr = BaseMstr;
        m_dmAlias = String.Empty;
        m_strDirectMethodPath = "App.direct" + m_dmAlias;
	}

    /// <summary>
    /// Constructor to create a new instance object for the class. 
    /// (with parameter for specifying DirectMethod alias)
    /// </summary>
    /// <param name="BaseMstr"></param>
    /// <param name="strDirectMethodAlias"></param>
    public CTxTree(BaseMaster BaseMstr, string strDirectMethodAlias)
    {
        m_BaseMstr = BaseMstr;
        m_dmAlias = "." + strDirectMethodAlias ;
        m_strDirectMethodPath = "App.direct" + m_dmAlias;
    }

    /// <summary>
    /// US:5967 US:5959
    /// Renders the side panel with the Summary and Current visit tabs.
    /// It also renders the patient treatment tree in the Summary tab
    /// </summary>
    /// <param name="pnlContainer"></param>
    public void RenderTxSidePanel(ext.Panel pnlContainer)
    {

        //create the new side panel container
        ext.Panel pnlNew = new ext.Panel()
        {
            Region = ext.Region.West, 
            Flex = 1,
            Border = false,
            Collapsible = true,
            Split = true,
            Header = false, 
            Frame= false, 
            Layout = "AnchorLayout"
        };

        //create the tab panel
        ext.TabPanel pnlTabPanel = new ext.TabPanel() 
        { 
            ID="tpTxTree",
            IDMode = ext.IDMode.Static,
            Layout="AnchorLayout",
            AnchorVertical = "100%",
            AnchorHorizontal = "100%",
            Border = false,
            Frame = false
        };

        //add the Summary and Current Visit tabs
        RenderInstrumentHx(pnlTabPanel);
        RenderVisitsPanel(pnlTabPanel);

        //add the tab panel to the new panel items collection
        pnlNew.Items.Add(pnlTabPanel);

        //add the new panel to the main panel container items collection
        pnlContainer.Items.Insert(0, pnlNew);

    }

    /// <summary>
    /// US:5959 US:5913
    /// Renders the Summary tab panel
    /// Task:6041
    /// </summary>
    /// <param name="tpContainer"></param>
    protected void RenderVisitsPanel(ext.TabPanel tpContainer)
    {
        //create the panel
        ext.Panel pnl = new ext.Panel()
        {
            ID = "pnlVisits",
            IDMode = ext.IDMode.Static,
            BodyPadding = 5,
            Title = "Visits",
            AnchorVertical = "100%",
            AnchorHorizontal = "100%", 
            Layout = "AnchorLayout",
            OverflowY = ext.Overflow.Auto
        };

        //render the tree
        RenderTxTree(pnl);

        //insert the new panel to the container tab panel
        tpContainer.Items.Add(pnl);
    }

    /// <summary>
    /// US:5959
    /// Renders the Current Visit tab panel
    /// Task:6041
    /// </summary>
    /// <param name="tpContainer"></param>
    protected void RenderInstrumentHx(ext.TabPanel tpContainer) 
    {
        //create the panel
        ext.Panel pnl = new ext.Panel()
        {
            ID = "pnlInstrumentHx",
            IDMode = ext.IDMode.Static,
            BodyPadding = 5,
            Title = "Instrument Hx"
        };

        //create panel items

        //ext.Panel pnlV1 = new ext.Panel()
        //{
        //    Layout = "HBoxLayout",
        //    Items = { 
        //        new ext.ComboBox
        //        {
        //            Flex = 3,
        //            BlankText = "Select visit type",
        //            ID = "cboVisitType",
        //            IDMode = ext.IDMode.Static
        //        },
        //        new ext.Button 
        //        {
        //            Text = "New Visit",
        //            Icon = ext.Icon.NoteAdd,
        //            ID = "btnAddVisit",
        //            IDMode = ext.IDMode.Static,
        //            MarginSpec = "0 0 0 10"
        //        }
        //    }
        //};

        //ext.Panel pnlV2 = new ext.Panel()
        //{
        //    Layout = "AnchorLayout",
        //    Items = {
        //        new ext.ComboBox
        //            {
        //                AnchorHorizontal = "100%",
        //                BlankText = "Visit Title",
        //                ID = "cboVisitTitle",
        //                IDMode = ext.IDMode.Static,
        //                MarginSpec = "10 0 15 0"
        //            }
        //    }
        //};

        
        //create add instrument button
        //ext.Button btn1 = new ext.Button() 
        //{ 
        //    Height=Unit.Pixel(25),
        //    Width = Unit.Pixel(140),
        //    ID="btnAddInstrument",
        //    Icon = ext.Icon.Magnifier,
        //    Text="Add Instrument",
        //    IDMode = ext.IDMode.Static,
        //    Handler= m_strDirectMethodPath + ".OnShowAddInstrument();"
        //};

        //add controls to the panel
        //pnl.Items.Add(pnlV1);
        //pnl.Items.Add(pnlV2);
        //pnl.Items.Add(btn1);

        //insert the new panel to the container tab panel
        tpContainer.Items.Add(pnl);
    }

    /// <summary>
    /// Renders the tree controls for the encounters, outcomes, instruments
    /// </summary>
    /// <param name="pnlContainer"></param>
    protected void RenderTxTree(ext.Panel pnlContainer)
    {

        //create the treatment panel
        ext.TreePanel tree = new ext.TreePanel()
        {
            ID = "tTxTree",
            IDMode = ext.IDMode.Static,
            AnchorHorizontal = "100%",
            AnchorVertical = "100%", 
            OverflowX = ext.Overflow.Auto, 
            OverflowY = ext.Overflow.Auto,
            ForceFit = true,
            RootVisible = false,
            Border = false,
            Frame = false,
            HideHeaders=true
        };

        //bind listeners to the tree nodes
        tree.Listeners.ItemClick.Handler = "TxTreeSelectedNode(record);";

        ext.Node nRoot = new ext.Node() { Expanded = true };
        tree.Root.Add(nRoot);

        //add nodes to the Root
        RenderCPANodes(nRoot);

        //auto expand first node 
        if (nRoot.Children.Count > 1)
        {
            nRoot.Children[1].Expanded = true;
        }

        //add the "apply pathway" button before the tree
        //RenderApplyPathwayButton(pnlContainer);

        //add the tree panel to the container panel
        pnlContainer.Items.Add(tree);
    }

    /* ------------------------------ */
    #region TreeNodes

    /// <summary>
    /// adds the "apply pathway" on top of the Treatment tree
    /// </summary>
    /// <param name="pnlParent"></param>
    protected void RenderApplyPathwayButton(ext.Panel pnlParent)
    {
        //create the button
        //add the button to the parent panel container
        pnlParent.Items.Add(new ext.Panel()
            {
                Items = { 
                    new ext.Button()
                    {
                        Text = "Apply Pathway",
                        Icon = ext.Icon.ApplicationAdd,
                        MarginSpec = "5 0 5 0",
                        Listeners =
                        {
                            Click = { 
                                Handler = "App.direct.PATSUMMARY.OnShowApplyPathway();"
                            }
                        }
                    }
                }
            });
    }
    
    /// <summary>
    /// US:5959 US:6253 US:6254
    /// Renders and adds the CPA node to the Treatment Tree
    /// </summary>
    /// <param name="nParentNode"></param>
    public void RenderCPANodes(ext.Node nParentNode) 
    {
        DataSet dsCPA = GetCPADS();
        if (dsCPA != null) {
            foreach (DataTable dt in dsCPA.Tables) {
                foreach (DataRow dr in dt.Rows) { 
                    
                    //create the CPA node
                    if (!dr.IsNull("CPA_TITLE"))
                    {
                        ext.Node nCPA = new ext.Node() { 
                            Text = dr["CPA_TITLE"].ToString(), 
                            EmptyChildren = true 
                        };

                        //render CPA patways
                        this.RenderPathwayNodes(Convert.ToInt32(dr["CPA_ID"]), nCPA);

                        nCPA.Icon = (nCPA.Children.Count > 0) ? ext.Icon.FolderPage : ext.Icon.Folder;

                        //add the PCA node to the parent node
                        nParentNode.Children.Add(nCPA);

                    }
                }
            }
        }
    }

    /// <summary>
    /// US:6253 US:6254
    /// Renders the pathway nodes for the parent CPA
    /// </summary>
    /// <param name="lCPAID"></param>
    /// <param name="nParentNode"></param>
    protected void RenderPathwayNodes(long lCPAID, ext.Node nParentNode) 
    {
        var ds = this.GetPathwaysDS(lCPAID);
        if (ds != null) { 
            foreach(DataTable dt in ds.Tables){
                foreach(DataRow dr in dt.Rows){
                    if (!dr.IsNull("PW_TITLE")) { 
                        //create the PATHWAY node
                        var nPW = new ext.Node()
                        {
                            Text = dr["PW_TITLE"].ToString(),
                            EmptyChildren = true,
                            CustomAttributes = { 
                                new ext.ConfigItem{
                                    Name = "pathway_id",
                                    Value = dr["PAT_PW_ID"].ToString(),
                                    Mode = ext.ParameterMode.Value
                                }
                            }
                        };

                        //render Pathway Events node
                        this.RenderPWEvents(Convert.ToInt32(dr["PAT_PW_ID"]), nPW);

                        nPW.Icon = (nPW.Children.Count > 0) ? ext.Icon.FolderPage : ext.Icon.Folder;

                        //add the pathways node to the parent node
                        nParentNode.Children.Add(nPW);
                    }
                }
            }
        }
    }

    /// <summary>
    /// US:6253 US:6254
    /// Renders the pathway events node
    /// </summary>
    /// <param name="lPWID"></param>
    /// <param name="nParentNode"></param>
    public void RenderPWEvents(long lPatPWID, ext.Node nParentNode) 
    {
        DataSet ds = this.GetPWEventsDS(lPatPWID);
        if(ds != null){
            foreach (DataTable dt in ds.Tables) {
                foreach (DataRow dr in dt.Rows) {
                    if (!dr.IsNull("EVENT_TITLE")) { 
                        
                        //create the event node
                        ext.Node nPWEvt = new ext.Node() { 
                            Text = dr["EVENT_TITLE"].ToString(),
                            EmptyChildren = true,
                            Expanded = true
                        };

                        //render event instruments node
                        this.RenderEventInstruments(lPatPWID, Convert.ToInt32(dr["PW_EVENT_ID"]), nPWEvt);

                        nPWEvt.Icon = (nPWEvt.Children.Count > 0) ? ext.Icon.FolderPage : ext.Icon.Folder;

                        //add event node to the parent node
                        nParentNode.Children.Add(nPWEvt);

                    }
                }
            }
        }
    }

    /// <summary>
    /// 
    /// </summary>
    /// <param name="strPWID"></param>
    /// <param name="nParentNode"></param>
    public void RenderPWEvents(string strPWID, ext.Node nParentNode)
    {
        DataSet ds = this.GetPWEventsDS(strPWID);
        if (ds != null)
        {
            foreach (DataTable dt in ds.Tables)
            {
                foreach (DataRow dr in dt.Rows)
                {
                    if (!dr.IsNull("EVENT_TITLE"))
                    {
                        //event status 2 = closed and tiu note submitted
                        long lEventStatus = Convert.ToInt32(dr["STATUS"]);
                     
                        //create the event node
                        ext.Node nPWEvt = new ext.Node()
                        {
                            Text = dr["EVENT_TITLE"].ToString(),
                            EmptyChildren = true,
                            Expanded = true,
                            CustomAttributes = { 
                                new ext.ConfigItem{
                                    Name = "is_event",
                                    Value = "true",
                                    Mode = ext.ParameterMode.Auto
                                },
                                new ext.ConfigItem{
                                    Name = "is_instrument",
                                    Value = "false",
                                    Mode = ext.ParameterMode.Auto
                                },
                                 new ext.ConfigItem{
                                    Name = "pat_pw_id",
                                    Value = dr["PAT_PW_ID"].ToString(),
                                    Mode = ext.ParameterMode.Auto
                                },
                                new ext.ConfigItem{
                                    Name = "pat_pw_event_id",
                                    Value = dr["PAT_PW_EVENT_ID"].ToString(),
                                    Mode = ext.ParameterMode.Auto
                                }
                            //    new ext.ConfigItem{
                             //       Name = "pw_id",
                               //     Value = dr["PW_ID"].ToString(),
                                 //   Mode = ext.ParameterMode.Auto
                               // },
                              //  new ext.ConfigItem{
                                //    Name = "pw_event_id",
                                  //  Value = dr["PW_EVENT_ID"].ToString(),
                                    //Mode = ext.ParameterMode.Auto
                                //}
                            }
                        };

                        //if event is closed don't show the edit options
                        if (lEventStatus == 2)
                        {
                            //nInstr.Icon = ext.Icon.Error;

                            //mark as not an event so the user 
                            //does not see the +/- tool bar options
                            //0 = is_event
                            nPWEvt.CustomAttributes[0].Value = "false";
                        }


                        //render event instruments node
                        this.RenderEventInstruments(Convert.ToInt32(dr["PAT_PW_ID"]), Convert.ToInt32(dr["PAT_PW_EVENT_ID"]), nPWEvt);

                        nPWEvt.Icon = (nPWEvt.Children.Count > 0) ? ext.Icon.FolderPage : ext.Icon.Folder;


                        //add event node to the parent node
                        nParentNode.Children.Add(nPWEvt);

                    }
                }
            }
        }
    }


    /// <summary>
    /// Renders nodes for all pathway events associated with a patient
    /// </summary>
    /// <param name="lPatPWID"></param>
    /// <param name="nParentNode"></param>
    public void RenderAllPWEvents(ext.Node nParentNode)
    {
        DataSet ds = this.GetPWEventsDS();
        if (ds != null)
        {
            foreach (DataTable dt in ds.Tables)
            {
                foreach (DataRow dr in dt.Rows)
                {
                    if (!dr.IsNull("EVENT_TITLE"))
                    {

                        //create the event node
                        ext.Node nPWEvt = new ext.Node()
                        {
                            Text = dr["EVENT_TITLE"].ToString(),
                            EmptyChildren = true, 
                            Expanded = true,
                            CustomAttributes = { 
                                new ext.ConfigItem{
                                    Name = "is_event",
                                    Value = "true",
                                    Mode = ext.ParameterMode.Auto
                                },
                                new ext.ConfigItem{
                                    Name = "is_instrument",
                                    Value = "false",
                                    Mode = ext.ParameterMode.Auto
                                }
                            }
                        };

                        //render event instruments node
                        this.RenderEventInstruments(Convert.ToInt32(dr["PAT_PW_ID"]), Convert.ToInt32(dr["PAT_PW_EVENT_ID"]), nPWEvt);

                        nPWEvt.Icon = (nPWEvt.Children.Count > 0) ? ext.Icon.FolderPage : ext.Icon.Folder;

                        //add event node to the parent node
                        nParentNode.Children.Add(nPWEvt);
                    }
                }
            }
        }
    }


    /// <summary>
    /// US:5861 US:6253 US:6254
    /// Render the event instruments node
    /// </summary>
    /// <param name="lPWID"></param>
    /// <param name="lEVTID"></param>
    /// <param name="nParentNode"></param>
    public void RenderEventInstruments(long lPWID, long lEVTID, ext.Node nParentNode) 
    {
        DataSet ds = this.GetEventInstruments(lPWID, lEVTID);
        if (ds != null) {
            foreach (DataTable dt in ds.Tables) {
                foreach (DataRow dr in dt.Rows) { 
                    if(!dr.IsNull("MODULE")){

                        //module name
                        string strModName = String.Empty;
                        if (!dr.IsNull("ACRONYM"))
                        {
                            strModName += dr["ACRONYM"].ToString() + " ";
                        }
                        strModName += dr["MODULE"].ToString();

                        //event status 2 = closed and tiu note submitted
                        long lEventStatus = Convert.ToInt32(dr["EVENT_STATUS"]);
                       
                        //create the instrument node
                        ext.Node nInstr = new ext.Node()
                        {
                           
                            Text = strModName,
                            Leaf = true,
                            EmptyChildren = true,
                            Icon = ext.Icon.Page,
                            CustomAttributes = { 
                                new ext.ConfigItem{
                                    Name = "is_event",
                                    Value = "false",
                                    Mode = ext.ParameterMode.Auto
                                },
                                new ext.ConfigItem{
                                    Name = "is_instrument",
                                    Value = "true",
                                    Mode = ext.ParameterMode.Auto
                                },
                                new ext.ConfigItem{
                                    Name = "patient_id",
                                    Value = m_BaseMstr.SelectedPatientID,
                                    Mode = ext.ParameterMode.Value
                                },
                                new ext.ConfigItem{
                                    Name = "item_type",
                                    Value = "evt_instrument",
                                    Mode = ext.ParameterMode.Value
                                },
                                new ext.ConfigItem{
                                    Name = "mid",
                                    Value = dr["MID"].ToString(),
                                    Mode = ext.ParameterMode.Value
                                },
                                new ext.ConfigItem{
                                    Name = "intake_id",
                                    Value = (!dr.IsNull("INTAKE_ID")) ? dr["INTAKE_ID"].ToString() : "0",
                                    Mode = ext.ParameterMode.Value
                                },
                                new ext.ConfigItem{
                                    Name = "status",
                                    Value = (!dr.IsNull("STATUS")) ? dr["STATUS"].ToString() : "0",
                                    Mode = ext.ParameterMode.Value
                                },
                                new ext.ConfigItem{
                                    Name = "region_id",
                                    Value = dr["region_id"].ToString(),
                                    Mode = ext.ParameterMode.Value
                                },
                                new ext.ConfigItem{
                                    Name = "site_id",
                                    Value = dr["site_id"].ToString(),
                                    Mode = ext.ParameterMode.Value
                                },
                              //  new ext.ConfigItem{
                              //      Name = "cpa_id",
                              //      Value = dr["cpa_id"].ToString(),
                              //      Mode = ext.ParameterMode.Value
                              //  },
                                new ext.ConfigItem{
                                    Name = "pat_pw_id",
                                    Value = dr["pat_pw_id"].ToString(),
                                    Mode = ext.ParameterMode.Value
                                },
                                new ext.ConfigItem{
                                    Name = "pat_pw_event_id",
                                    Value = dr["pat_pw_event_id"].ToString(),
                                    Mode = ext.ParameterMode.Value
                                },
                                new ext.ConfigItem{
                                    Name = "pw_event_module_id",
                                    Value = dr["pw_event_module_id"].ToString(),
                                    Mode = ext.ParameterMode.Value
                                }
                            }
                        };

                        //if module completed mark it with a check
                        if (Convert.ToInt32(dr["STATUS"]) == 1)
                        {
                            nInstr.Qtip = "Completed Instrument";
                            nInstr.Icon = ext.Icon.Tick;                       
                        }   
                        else if (Convert.ToInt32(dr["STATUS"]) == 2)
                        {
                            nInstr.Qtip = "Locked Instrument";
                            nInstr.Icon = ext.Icon.Lock;
                        }
                        else
                        {
                            //if event is closed and instrument was not completed change icon to error
                            if (lEventStatus == 2)
                            {
                                nInstr.Icon = ext.Icon.Error;

                                //mark as not an instrument so the user 
                                //can not edit an instrument on a closed event
                                //1 = is_instrument
                                nInstr.CustomAttributes[1].Value = "false";
                                //3 = item type
                                nInstr.CustomAttributes[3].Value = "evt_closed";
                            }
                        }

                        //add the instrument node to the parent node
                        nParentNode.Children.Add(nInstr);
                    }
                }
            }
        }
    }

    #endregion

    /* ------------------------------ */
    #region GetDataSets

    /// <summary>
    /// Gets the dataset to build the Clinical Settings node
    /// </summary>
    /// <returns></returns>
    protected DataSet GetCPADS()
    {
        CCPA ccpa = new CCPA();
        var ds = ccpa.GetCPADS(m_BaseMstr, m_BaseMstr.RegionID, m_BaseMstr.SiteID);
        return ds;
    }

    /// <summary>
    /// return a dataset of pathway records for the CPA
    /// </summary>
    /// <param name="lCPAID"></param>
    /// <returns></returns>
    protected DataSet GetPathwaysDS(long lCPAID) 
    {
        CCPA ccpa = new CCPA();
        //var ds = ccpa.GetPatPWDS(m_BaseMstr, m_BaseMstr.SelectedPatientID, lCPAID);
        DataSet ds = null;
        return ds;
    }

    /// <summary>
    /// Gets the dataset to build the Pathway Events node
    /// </summary>
    /// <returns></returns>
    protected DataSet GetPWEventsDS(long lPWID)
    {
        CCPA ccpa = new CCPA();
        var ds = ccpa.GetPatPWEventDS(m_BaseMstr, m_BaseMstr.SelectedPatientID, lPWID);
        return ds;
    }

    protected DataSet GetPWEventsDS(string strPWList)
    {
        CCPA ccpa = new CCPA();
        var dsOriginal = ccpa.GetPatPWEventDS(m_BaseMstr, m_BaseMstr.SelectedPatientID);

        if (strPWList.Length > 1)
        {
            strPWList = "," + strPWList;
            DataSet dsNew = null;

            if (dsOriginal != null)
            {
                if (dsOriginal.Tables.Count > 0)
                {
                    dsNew = new DataSet()
                    {
                        Tables = { 
                            dsOriginal.Tables[0].Clone()
                        }
                    };

                    foreach (DataRow dr in dsOriginal.Tables[0].Rows)
                    {
                        if (!dr.IsNull("PAT_PW_ID"))
                        {
                            string strPatPWID = "," + dr["PAT_PW_ID"].ToString() + ",";
                            if (strPWList.IndexOf(strPatPWID) > -1)
                            {
                                DataRow drNew = dsNew.Tables[0].NewRow();
                                for (int a = 0; a < dsNew.Tables[0].Columns.Count; a++)
                                {
                                    drNew[a] = dr[a];
                                }
                                dsNew.Tables[0].Rows.Add(drNew);
                            }
                        }
                    }
                    dsNew.Tables[0].AcceptChanges();
                    dsNew.AcceptChanges();
                }
            }
            return dsNew;
        }

        return dsOriginal;

    }

    /// <summary>
    /// Gets dataset of all pathway events associated with a patient
    /// </summary>
    /// <param name="lPWID"></param>
    /// <returns></returns>
    protected DataSet GetPWEventsDS()
    {
        CCPA ccpa = new CCPA();
        var ds = ccpa.GetPatPWEventDS(m_BaseMstr, m_BaseMstr.SelectedPatientID);
        return ds;
    }

    /// <summary>
    /// US:5861
    /// Gets the dataset to build the Pathway Event Instruments node
    /// </summary>
    /// <param name="lEVTID"></param>
    /// <returns></returns>
    protected DataSet GetEventInstruments(long lPWID, long lEVTID) 
    {
        CCPA ccpa = new CCPA();
        var ds = ccpa.GetPatPWEventModuleDS(m_BaseMstr, m_BaseMstr.SelectedPatientID, lPWID, lEVTID);
        return ds;
    }

    #endregion
}